#include <QCoreApplication>
#include <cmath>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

extern "C"
{
    #include "usblismpi26.h"
}

int main(int argc, char *argv[])
{
    QCoreApplication a(argc, argv);

    quint16 uval16;
    quint8 uval8;
    qint32 err;
    double fval;
    double AFE_Scale;
    quint16 wMult;
    quint16 wDiv;

    setbuf(stdout, NULL);
    printf("\n\n******* USB LISM-PI26xx - ADC Settings *******\n\n");

    ls_initialize(1024*1024,4096);

    if (ls_enumdevices() == 0) {
        printf("No devices available.\n");
        return 1;
    }
    printf("Number of devices available: %d\n", ls_enumdevices());

    // Open device with index = 0
    printf("Open device: ");
    err = ls_opendevicebyindex(0);
    if (err == RET_OK) {
        printf("%s - ",ls_getproductname(ls_currentdeviceindex()));
        printf("%s\n\n",ls_getserialnumber(ls_currentdeviceindex()));

        printf("Calculate AFE_Scale:\n");
        AFE_Scale = 1;
        printf("ADC_GAIN_MULT: ");
        err = ls_getgainmult(LISM_ADDR, wMult, 1000);
        if (err == RET_OK) {
            printf("  %d\n",wMult);
            printf("ADC_GAIN_DIV: ");
            err = ls_getgaindiv(LISM_ADDR, wDiv,1000);
            if (err == RET_OK) {
                printf("   %d\n",wDiv);
                AFE_Scale = double(wMult)/double(wDiv);
            } else {
                printf("ERROR: %s\n",ls_geterrorstring(err));
            }
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }
        printf("AFE_Scale:       %.4f\n\n",AFE_Scale);

        printf("Set ADC Parameters:\n");
        //Set VRef (Full Scale) to 1.6V
        printf("Set ADC VRef: ");
        err = ls_setadcvref(LISM_ADDR, ADC_VREF_1V6, 1000);
        if (err == RET_OK) {
            printf("   1.6 V\n");
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }

        // Set ADC Gain to 2V/V
        // Gain = (76 / (76 - uval8)) * AFE_Scale; val8 = 0..63
        // => uval8 = 76 - ((76 * AFE_SCALE)/Gain)
        fval = 2.0;
        uval8 = 76.0 - ((76.0 * AFE_Scale)/fval);
        printf("Set ADC Gain: ");
        err = ls_setadcgain(LISM_ADDR, uval8,1000);
        if (err == RET_OK) {
            printf("   %.4f V/V\n", fval);
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }

        // Set ADC Offset to -10 mV
        // Offset[mv] = uval16 * (250mV / 255); uval16 = 0..255
        // => uval16 = abs(Offset) / (250 / 255)
        // For negative offset Bit[8] must be set to 1, else 0.
        // e.g. +11mV => uval16 = 11 / (250 / 255) = 11.2
        // e.g. -11mV => uval16 = (11 / (250 / 255)) | 0x0100;
        fval = -10;
        uval16 = (fabs(fval) / (250.0/255.0));
        uval16 |= 0x0100;
        printf("Set ADC Offset: ");
        err = ls_setadcoffset(LISM_ADDR, uval16,1000);
        if (err == RET_OK) {
            printf("%.4f mV\n",fval);
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }

        // Set ADC Bias to -500 mV
        // Bias[mV] = (uval16 / 1023) * 2500mV ; uval16 = 0..1023
        // => uval16 = (abs(Bias) * 1023 ) / 2500
        fval = -500;
        uval16 = (fabs(fval) * 1023.0) / 2500.0;
        printf("Set ADC Bias: ");
        err = ls_setadcbias(LISM_ADDR, uval16,1000);
        if (err == RET_OK) {
            printf("  %.4f mV\n",fval);
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }

        printf("\nGet ADC Parameters:\n");

        //Get VRef (Full Scale)
        printf("Get ADC VRef: ");
        err = ls_getadcvref(LISM_ADDR, uval8,1000);
        if (err == RET_OK) {
            if (uval8 == ADC_VREF_1V6) {
                printf("   1.6 V\n");
            } else {
                printf("   2.0 V\n");
            }
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }

        //Get ADC Gain
        printf("Get ADC Gain: ");
        err = ls_getadcgain(LISM_ADDR, uval8,1000);
        if (err == RET_OK) {
            fval = (76.0 / (76.0 - uval8)) * AFE_Scale;
            printf("   %f V/V\n",fval);
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }

        // Get ADC Offset
        printf("Get ADC Offset: ");
        err = ls_getadcoffset(LISM_ADDR, uval16,1000);
        if (err == RET_OK) {
            fval = (uval16 & 0x00FF) * (250.0 / 255.0);
            if ((uval16 & 0x0100) == 0x0100) {
                 fval *= -1;
            }
            printf("%.4f mv\n",fval);
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }

        // Get ADC Bias
        printf("Get ADC Bias: ");
        err = ls_getadcbias(LISM_ADDR, uval16,1000);
        if (err == RET_OK) {
            fval = (uval16 / 1023.0) * 2500 * (-1);
            printf("  %.4f mV\n",fval);
        } else {
            printf("ERROR: %s\n\n",ls_geterrorstring(err));
        }

        // Close device
        ls_closedevice();
        printf("\nDevice closed\n");
    } else {
        printf("ERROR: %s\n",ls_geterrorstring(err));
    }

    return 0;
}
