#include <QCoreApplication>
#include <cmath>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

extern "C"
{
    #include "usblismpi26.h"
}

int main(int argc, char *argv[])
{
    QCoreApplication a(argc, argv);

    quint8  IfConfig;
    qint32  err;
    double  Tpclk;
    quint32 STcycles;
    quint32 SThigh;
    quint32 STlow;
    quint32 LineRate;
    quint32 IntegrationTime;

   /*
    * CMOS Image Linear Sensor S11639-01
    * Pixel = 2048
    * Edge Delay = 89
    * ADC Output Delay = 9
    * Frame Valid = 5 trailing cycles
    *
    * ST(Period) >= 2048+ 89 + 9 + 5 = 2151 Cycles
    * Min. ST(LOW) >= 100 Cylces
    * Min. ST(High) >= 6 Cycles
    * Integration Time = ST(High) + 48 Cycles
    *
    * Min. Data PacketLength = 2 * 2048 pixel = 4096 Bytes
    */

    setbuf(stdout, NULL);
    printf("\n\n******* USB LISM-PI26xx - Integration Time *******\n\n");

    ls_initialize(1024*1024,4096);

    if (ls_enumdevices() == 0) {
        printf("No devices available.\n");
        return 1;
    }
    printf("\nNumber of devices available: %d\n", ls_enumdevices());

    // Open device with index = 0
    printf("Open device: ");
    err = ls_opendevicebyindex(0);
    if (err == RET_OK) {
        printf("%s - ",ls_getproductname(ls_currentdeviceindex()));
        printf("%s\n\n",ls_getserialnumber(ls_currentdeviceindex()));

        IntegrationTime = 500; // 500µs
        printf("Set the integration time to %d us and max. possible line rate:\n",IntegrationTime);
        printf("T(Int.)  = %d us\n", IntegrationTime);

        //Ready DATA_IF_CONFIG Register to determine pixel frequency / period Tpclk
        Tpclk = 1.0/10000000.0;
        err = ls_getifconfig(LISM_ADDR, IfConfig, 1000);
        if (err != RET_OK) {
            printf("Get IF Config: ERROR: %s\n", ls_geterrorstring(err));
        } else {
            switch (IfConfig & 0x07) {
            case 0x00: Tpclk = 1.0/200000.0; break;
            case 0x01: Tpclk = 1.0/250000.0; break;
            case 0x02: Tpclk = 1.0/400000.0; break;
            case 0x03: Tpclk = 1.0/500000.0; break;
            case 0x04: Tpclk = 1.0/1000000.0; break;
            case 0x05: Tpclk = 1.0/2000000.0; break;
            case 0x06: Tpclk = 1.0/5000000.0; break;
            case 0x07: Tpclk = 1.0/10000000.0; break;
            }
            printf("T(pclk)  = %.0f ns\n", (Tpclk * 1000000000 ));
        }

        //Calculate ST(Period) for an integration time of T(Int.) = 500µs
        //Conditions:
        //  1. ST(High) >= 6
        //  2. ST(Low) >= 100
        //  3. ST(Period) = ST(High) + ST(Low) >= 2151
        //ST(High) = (T(Int.) / Tpclk) - 48 >= 6

        SThigh = ((IntegrationTime * 1000.0) / (Tpclk * 1000000000.0));
        if (SThigh >= (48+6)) {
            SThigh -= 48;
        } else {
            SThigh = 6;
            printf("Integratione time is less than minimum - set to minmum:\n");
            printf("T(Int.)  = %.0f us\n", ((SThigh + 48.0) * Tpclk * 1000000.0));
        }
        if ((2151.0 - SThigh) < 100) {
            STlow = 100;
        } else {
            STlow = 2151 - SThigh;
        }
        STcycles = SThigh + STlow;
        LineRate = (1.0/STcycles) / Tpclk;
        printf("STcycles = %d Cycles\n", STcycles);
        err = ls_setstpulse(LISM_ADDR, SThigh, STlow, 1000);
        if (err == RET_OK) {
            printf("ST(High) = %d Cycles\n", SThigh);
            printf("ST(Low)  = %d Cycles\n", STlow);
            printf("f(Line)  = %d Hz\n", LineRate);
        } else {
            printf("ERROR: %s\n",ls_geterrorstring(err));
        }
        // Close device
        ls_closedevice();
        printf("\nDevice closed\n");
    } else {
        printf("ERROR: %s\n",ls_geterrorstring(err));
    }

    return 0;
}
