#include <QCoreApplication>
#include <cmath>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

extern "C"
{
    #include "usblismpi26.h"
}

int main(int argc, char *argv[])
{
    QCoreApplication a(argc, argv);

    quint8 IfConfig;
    qint32 err;
    double Tpclk;
    quint32 STcycles;
    quint32 SThigh;
    quint32 STlow;
    quint32 LineRate;

   /*
    * CMOS Image Linear Sensor S11639-01
    * Pixel = 2048
    * Edge Delay = 89
    * ADC Output Delay = 9
    * Frame Valid = 5 trailing cycles
    *
    * ST(Period) >= 2048+ 89 + 9 + 5 = 2151 Cycles
    * Min. ST(LOW) >= 100 Cylces
    * Min. ST(High) >= 6 Cycles
    * Integration Time = ST(High) + 48 Cycles
    *
    * Min. Data PacketLength = 2 * 2048 pixel = 4096 Bytes
    */

    setbuf(stdout, NULL);
    printf("\n\n******* USB LISM-PI26xx - Line Rate *******\n\n");

    ls_initialize(1024*1024,4096);

    if (ls_enumdevices() == 0) {
        printf("No devices available.\n");
        return 1;
    }
    printf("Number of devices available: %d\n", ls_enumdevices());

    // Open device with index = 0
    printf("Open device: ");
    err = ls_opendevicebyindex(0);
    if (err == RET_OK) {
        printf("%s - ",ls_getproductname(ls_currentdeviceindex()));
        printf("%s\n\n",ls_getserialnumber(ls_currentdeviceindex()));

        LineRate = 2000; //2kHz
        printf("Set the line rate to %d lines/second and max. possible integration time:\n", LineRate);
        printf("f(Line)  = %d Hz\n", LineRate);

        //Ready DATA_IF_CONFIG Register to determine pixel frequency / period Tpclk
        Tpclk = 1.0/10000000.0;
        err = ls_getifconfig(LISM_ADDR, IfConfig, 1000);
        if (err != RET_OK) {
            printf("Get IF Config: ERROR: %s", ls_geterrorstring(err));
        } else {
            switch (IfConfig & 0x07) {
            case 0x00: Tpclk = 1.0/200000.0; break;
            case 0x01: Tpclk = 1.0/250000.0; break;
            case 0x02: Tpclk = 1.0/400000.0; break;
            case 0x03: Tpclk = 1.0/500000.0; break;
            case 0x04: Tpclk = 1.0/1000000.0; break;
            case 0x05: Tpclk = 1.0/2000000.0; break;
            case 0x06: Tpclk = 1.0/5000000.0; break;
            case 0x07: Tpclk = 1.0/10000000.0; break;
            }
            printf("T(pclk)  = %.0f ns\n", (Tpclk * 1000000000 ));
        }

        //Calculate ST(Period) for a frequency of f = 2kHz
        //ST(Period) = 1/f = 1/2000 = 500 µs
        //ST(Cycles/Period) = 500µs / Tpclk

        STcycles = (1.0/LineRate) / Tpclk;
        printf("STcycles = %d Cycles\n", STcycles);
        if (STcycles > 2150) {
            STlow = 100; // Min. ST(Low)
            SThigh = STcycles - STlow;
            err = ls_setstpulse(LISM_ADDR, SThigh, STlow, 1000);
            if (err == RET_OK) {
                printf("ST(High) = %d Cycles\n", SThigh);
                printf("ST(Low)  = %d Cycles\n", STlow);
                printf("T(Int.)  = %.0f us\n", ((SThigh + 48.0) * Tpclk * 1000000.0));
            } else {
                printf("ERROR: %s\n",ls_geterrorstring(err));
            }
        } else {
            printf("STcycles is less than minimum\n");
        }

        // Close device
        ls_closedevice();
        printf("\nDevice closed\n");
    } else {
        printf("ERROR: %s\n",ls_geterrorstring(err));
    }

    return 0;
}
