#include <QCoreApplication>
#include <QFile>
#include <QDateTime>
#include <QThread>
#include <QVector>
#include <QtGlobal>

#include <cstdio>
#include <cstdlib>
#include <cstdint>
#include <cstring>
#include <stdexcept>
#include <iostream>

extern "C"
{
    #include "usblismpi26.h"
}

// -----------------------------------------------------------------------------
// TIFF-Structure & Help functions
// -----------------------------------------------------------------------------
#pragma pack(push, 1)
struct TIFDEntry {
    quint16 Tag;
    quint16 FieldType;
    quint32 Count;
    quint32 ValueOffset;
};
#pragma pack(pop)

static void setIFD(TIFDEntry &e,
                   quint16 tag,
                   quint16 type,
                   quint32 count,
                   quint32 value)
{
    e.Tag         = tag;
    e.FieldType   = type;
    e.Count       = count;
    e.ValueOffset = value;
}

// "YYYY:MM:DD HH:MM:SS\0" (20 Byte)
static void buildTIFFDateTime20(char out20[20])
{
    const QString dt = QDateTime::currentDateTime()
                           .toString(QStringLiteral("yyyy:MM:dd HH:mm:ss"));
    QByteArray ascii = dt.toLatin1();

    std::memset(out20, 0, 20);
    const int n = qMin(19, ascii.size());
    for (int i = 0; i < n; ++i)
        out20[i] = ascii.at(i);
    out20[19] = '\0';
}

// Speichert ein 16-Bit-Graustufenbild als unkomprimiertes TIFF.
// imageDataLE: width * height 16-Bit (Little Endian, wie vom Sensor)
static void saveTIFF_Qt(const QString &fileName,
                        const quint16 *imageDataLE,
                        quint16 width,
                        quint16 height,
                        int dpi = 300)
{
    if (!imageDataLE || width == 0 || height == 0)
        throw std::runtime_error("Invalid arguments for saveTIFF_Qt");

    QFile file(fileName);
    if (!file.open(QIODevice::WriteOnly)) {
        throw std::runtime_error("Cannot open TIFF output file.");
    }

    const quint32 NUM_TAGS = 13;
    const quint16 PHOTOMETRIC_MINISBLACK = 1;
    const quint16 RESOLUTION_UNIT_INCH   = 2;
    const quint16 BITS_PER_SAMPLE        = 16;
    const quint16 SAMPLES_PER_PIXEL      = 1;

    // TIFF-Header ("II", 42, IFD-Offset = 8)
    char ii[2] = { 'I', 'I' };
    quint16 magic    = 42;
    quint32 ifdOffset = 8;

    file.write(ii, 2);
    file.write(reinterpret_cast<const char*>(&magic), sizeof(magic));
    file.write(reinterpret_cast<const char*>(&ifdOffset), sizeof(ifdOffset));

    // Offsets berechnen
    const quint32 ifdSize        = 2 + NUM_TAGS * sizeof(TIFDEntry) + 4;
    const quint32 resOffset      = ifdOffset + ifdSize;      // 2 * RATIONAL
    const quint32 dateTimeOffset = resOffset + 16;           // 20 Byte ASCII
    const quint32 imageOffset    = dateTimeOffset + 20;      // Bilddaten
    const quint32 imageBytes     =
            static_cast<quint32>(width) *
            static_cast<quint32>(height) * 2u;

    // IFD-Einträge
    TIFDEntry entries[NUM_TAGS];
    quint32 i = 0;
    setIFD(entries[i++], 256, 4, 1, static_cast<quint32>(width));   // ImageWidth
    setIFD(entries[i++], 257, 4, 1, static_cast<quint32>(height));  // ImageLength
    setIFD(entries[i++], 258, 3, 1, BITS_PER_SAMPLE);               // BitsPerSample
    setIFD(entries[i++], 259, 3, 1, 1);                             // Compression = none
    setIFD(entries[i++], 262, 3, 1, PHOTOMETRIC_MINISBLACK);        // Photometric
    setIFD(entries[i++], 273, 4, 1, imageOffset);                   // StripOffsets
    setIFD(entries[i++], 277, 3, 1, SAMPLES_PER_PIXEL);             // SamplesPerPixel
    setIFD(entries[i++], 278, 4, 1, static_cast<quint32>(height));  // RowsPerStrip
    setIFD(entries[i++], 279, 4, 1, imageBytes);                    // StripByteCounts
    setIFD(entries[i++], 282, 5, 1, resOffset);                     // XResolution
    setIFD(entries[i++], 283, 5, 1, resOffset + 8);                 // YResolution
    setIFD(entries[i++], 296, 3, 1, RESOLUTION_UNIT_INCH);          // ResolutionUnit
    setIFD(entries[i++], 306, 2, 20, dateTimeOffset);               // DateTime

    if (i != NUM_TAGS) {
        throw std::runtime_error("IFD tag count mismatch.");
    }

    quint16 tagCount = static_cast<quint16>(NUM_TAGS);
    quint32 nextIFD  = 0;

    file.write(reinterpret_cast<const char*>(&tagCount), sizeof(tagCount));
    file.write(reinterpret_cast<const char*>(entries), sizeof(entries));
    file.write(reinterpret_cast<const char*>(&nextIFD), sizeof(nextIFD));

    // X/Y-Resolution (dpi/1)
    quint32 xres[2] = { static_cast<quint32>(dpi), 1u };
    quint32 yres[2] = { static_cast<quint32>(dpi), 1u };
    file.write(reinterpret_cast<const char*>(xres), sizeof(xres));
    file.write(reinterpret_cast<const char*>(yres), sizeof(yres));

    // DateTime (20 Byte)
    char dt20[20];
    buildTIFFDateTime20(dt20);
    file.write(dt20, 20);

    // Bilddaten (bereits Little Endian)
    file.write(reinterpret_cast<const char*>(imageDataLE), imageBytes);

    file.close();
}

// -----------------------------------------------------------------------------
// Camera help functions
// -----------------------------------------------------------------------------
static void clearCameraBuffer(quint8 *dst)
{
    qint32 err;
    qint32 available = 0;
    qint32 cleared = 0;

    printf("Stop Camera...");
    err = ls_setstate(LISM_ADDR, 0, 1000);
    if (err == RET_OK) {
        printf("OK\n");
    } else {
        printf("Error: %s\n", ls_geterrorstring(err));
    }

    QThread::msleep(100);

    printf("Clearing Buffer...");
    available = ls_getpipe(dst, 0);
    while (available > 0) {
        available = ls_getpipe(dst, available);
        if (available > 0) {
            cleared += available;
            available = ls_getpipe(dst, available);
        }
    }
    printf("%d bytes cleared\n",cleared);
}

static bool readCameraFrame(quint8 *dst, quint16 width, quint16 height)
{
    qint32 bytesread;
    qint32 available;
    const quint32 imageSize = static_cast<quint32>(width) *
                              static_cast<quint32>(height) * 2u;

    // discard the first 4 lines
    const quint32 discardBytes = static_cast<quint32>(width) * 4u * 2u;
    QVector<quint8> discard(discardBytes);

    printf("discard lines:...");
    ls_waitforpipecount(static_cast<qint32>(discardBytes), available, 10000);
    while(available < static_cast<qint32>(discardBytes)) {
        ls_waitforpipecount(static_cast<qint32>(discardBytes), available, 10000);
    }

    bytesread = ls_getpipe(discard.data(), static_cast<qint32>(discardBytes));
    printf("%d bytes\n", bytesread);

    printf("Reading Buffer...");
    ls_waitforpipecount(static_cast<qint32>(imageSize), available, 10000);
    while(available < static_cast<qint32>(imageSize)) {
        ls_waitforpipecount(static_cast<qint32>(imageSize), available, 10000);
    }
    bytesread = ls_getpipe(dst, static_cast<qint32>(imageSize));

    printf("%u bytes read\n", bytesread);
    return (bytesread == static_cast<qint32>(imageSize));
}

int main(int argc, char *argv[])
{
    QCoreApplication app(argc, argv);
    setbuf(stdout, nullptr);

    printf("\n\n******* USB LISM-PI26xx - Save Frame to TIFF *******\n\n");

    const quint16 W = 2048;
    const quint16 H = 2048;
    const quint32 imageSize = static_cast<quint32>(W) *
                              static_cast<quint32>(H) * 2u;

    QVector<quint8> raw(static_cast<int>(imageSize));

    ls_initialize(imageSize, static_cast<qint32>(W * 2));

    if (ls_enumdevices() == 0) {
        printf("No devices available.\n");
        return 1;
    }

    printf("Number of devices available: %d\n", ls_enumdevices());

    printf("Open device: ");
    qint32 err = ls_opendevicebyindex(0);
    if (err != RET_OK) {
        printf("ERROR: %s\n", ls_geterrorstring(err));
        return 1;
    }

    std::printf("%s - ", ls_getproductname(ls_currentdeviceindex()));
    std::printf("%s\n\n", ls_getserialnumber(ls_currentdeviceindex()));

    clearCameraBuffer(raw.data());

    qint32 packetLength = 0;
    printf("\nGet PacketLength (Device): ");
    err = ls_getpacketlength(packetLength, 1000);
    if (err != RET_OK) {
        printf("ERROR: %s\n", ls_geterrorstring(err));
    } else {
        printf("%u Bytes\n", packetLength);
        printf("Close device to set USB PacketLength\n");
        ls_closedevice();

        printf("Set PacketLength (USB): ");
        err = ls_setpacketlength(packetLength);
        if (err != RET_OK) {
            printf("ERROR: %s\n", ls_geterrorstring(err));
        } else {
            printf("%u Bytes\n", packetLength);
            printf("(Re)Open device\n\n");
            err = ls_opendevicebyindex(0);
            if (err != RET_OK) {
                printf("ERROR: %s\n", ls_geterrorstring(err));
            }
        }
    }

    if (ls_currentdeviceindex() < 0) {
        printf("No current device.\n");
        return 1;
    }

    printf("Start Camera...");
    err = ls_setstate(LISM_ADDR, 1, 1000);
    if (err != RET_OK) {
        printf("Error: %s\n", ls_geterrorstring(err));
        ls_closedevice();
        printf("\nDevice closed\n");
        return 1;
    }
    std::printf("OK\n");

    try {
        if (!readCameraFrame(raw.data(), W, H)) {
            throw std::runtime_error("ls_getpipe failed / incomplete frame.");
        }

        // Rohdaten als 16-Bit-Pointer interpretieren
        quint16 *px = reinterpret_cast<quint16*>(raw.data());
        QString path = QCoreApplication::applicationDirPath()+ "/frame.tiff";
        saveTIFF_Qt(path, px, W, H, 300);
        printf("<frame.tiff> created\n");

        clearCameraBuffer(raw.data());
        ls_closedevice();
        printf("\nDevice closed\n");
    }
    catch (const std::exception &ex) {
        fprintf(stderr, "ERROR: %s\n", ex.what());
        ls_closedevice();
        return 1;
    }

    return 0;
}
